"""
Dual-Axis Tracking
==================

Example of a custom Mount class.
"""

# %%
# Dual-axis trackers can track the sun in two dimensions across the sky dome
# instead of just one like single-axis trackers.  This example shows how to
# model a simple dual-axis tracking system using ModelChain with a custom
# Mount class.

from pvlib import pvsystem, location, modelchain
import pandas as pd
import matplotlib.pyplot as plt

# %%
# New Mount classes should extend ``pvlib.pvsystem.AbstractMount``
# and must implement a ``get_orientation(solar_zenith, solar_azimuth)`` method:


class DualAxisTrackerMount(pvsystem.AbstractMount):
    def get_orientation(self, solar_zenith, solar_azimuth):
        # no rotation limits, no backtracking
        return {'surface_tilt': solar_zenith, 'surface_azimuth': solar_azimuth}


loc = location.Location(40, -80)
array = pvsystem.Array(
    mount=DualAxisTrackerMount(),
    module_parameters=dict(pdc0=1, gamma_pdc=-0.004, b=0.05),
    temperature_model_parameters=dict(a=-3.56, b=-0.075, deltaT=3))
system = pvsystem.PVSystem(arrays=[array], inverter_parameters=dict(pdc0=3))
mc = modelchain.ModelChain(system, loc, spectral_model='no_loss')

times = pd.date_range('2019-01-01 06:00', '2019-01-01 18:00', freq='5min',
                      tz='Etc/GMT+5')
weather = loc.get_clearsky(times)
mc.run_model(weather)

mc.results.ac.plot()
plt.ylabel('Output Power')
plt.show()
